<?php
//------------------------------------------------------------------------------
// OptiAdmin :: Databas-klassen (backup)
//------------------------------------------------------------------------------
abstract class OptiDatabase
{
	var $tbls = array();
}
//------------------------------------------------------------------------------
// OptiAdmin :: Tabell-klassen (list, export)
//------------------------------------------------------------------------------
abstract class OptiTable
{
	//data
	var $rows = array();
	var $id_list = array();
	var $antal = 0;
	var $found_rows = 0;
	var $filter_search_message = '';

	//params
	var $url = '';
	var $limit = 50;					//sparas i SESSION
	var $offset = 0;
	var $sort = 'id';					//sparas i SESSION
	var $order = 'ASC';				//sparas i SESSION
	var $search = NULL;				//sparas i SESSION
	var $filter = array();		//sparas i SESSION

	//settings
	var $sort_cols = array();
	var $search_cols = array();
	var $filter_cols = array();

	//----------------------------------------------------------------------------
	// LIST: Hämtar parametrar för sortering, ordning, sök och filtrering
	// Hämtas från: 1) $_GET, 2) $_SESSION, 3) default (i den ordningen)
	// Parametrarna sparas sedan i $user->data
	//----------------------------------------------------------------------------
	public function load_params($module)
	{
		global $cfg, $user;

		//läser från $user->data
		$this->limit = ( isset($user->data[$module]['limit']) ) ? $user->data[$module]['limit'] : $cfg->get('mysql_limit');
		$this->sort = ( isset($user->data[$module]['sort']) ) ? $user->data[$module]['sort'] : $this->sort;
		$this->order = ( isset($user->data[$module]['order']) ) ? $user->data[$module]['order'] : $this->order;
		$this->search = ( isset($user->data[$module]['search']) ) ? $user->data[$module]['search'] : $this->search;
		$this->filter = ( isset($user->data[$module]['filter']) ) ? $user->data[$module]['filter'] : $this->filter;

		//LIMIT
		if ( isset($_GET['limit']) )
		{
			$this->limit = intval($_GET['limit']);
		}
		//OFFSET
		if ( isset($_GET['offset']) )
		{
			$this->offset = intval($_GET['offset']);
		}
		//SORT - tillåt bara kolumn som går att sortera på
		if ( isset($_GET['sort']) && in_array($_GET['sort'], $this->sort_cols) )
		{
			$this->sort = clean($_GET['sort']);
		}
		//ORDER - tillåt bara ASC eller DESC
		if ( isset($_GET['order']) )
		{
			$this->order = ( $_GET['order'] == 'desc' ) ? 'DESC' : 'ASC';
		}
		//SEARCH
		if ( isset($_GET['search']) )
		{
			$this->search = clean($_GET['search']);
		}
		//FILTER - hämta bara kolumner som går att filtrera på
		foreach ($this->filter_cols as $col)
		{
			if ( isset($_GET[$col]) )
			{
				$this->filter[$col] = $_GET[$col];
			}
		}

		//sparar i $user->data
		$user->data[$module]['limit'] = ( $this->limit > 0 ) ? $this->limit : $cfg->get('mysql_limit'); //spara ej limit=0 (visa alla)
		$user->data[$module]['sort'] = $this->sort;
		$user->data[$module]['order'] = $this->order;
		$user->data[$module]['search'] = $this->search;
		$user->data[$module]['filter'] = $this->filter;
	}

	//----------------------------------------------------------------------------
	// LIST: Genererar SQL-kod (col = LIKE '%search%')
	//----------------------------------------------------------------------------
	protected function sql_search($prefix)
	{
		$return = '';
		if ( $this->search )
		{
			$sql_search = array();
			foreach ($this->search_cols as $col)
			{
				$sql_search[] = $prefix.".".$col." LIKE '%".$this->search."%'";
   		}
			$return = "(".implode(" OR ", $sql_search).")";
		}
		return $return;
	}

	//----------------------------------------------------------------------------
	// LIST: Hämtar totala antalet rader
	//----------------------------------------------------------------------------
	protected function sql_found_rows()
	{
		//om antalet rader har begränsats av limit: hämtar det totala antalet rader
		if ( $this->antal == $this->limit || $this->offset !== 0 )
		{
			$info = result(query("SELECT FOUND_ROWS() AS num_results"), false);
			$this->found_rows = intval($info['num_results']);
		}
		else
		{
			$this->found_rows = $this->antal;
		}
	}

	//----------------------------------------------------------------------------
	// LIST: Skapar sorteringslänk, ex: <a href="index.php?module=[...]&sort=[...]&order=desc" class="asc">[...]</a>
	//----------------------------------------------------------------------------
	public function create_sortable($col_name, $caption)
	{
		$url = $this->url;
		$url .= ( strpos($url, '?') === false ) ? '?' : '&';
		$url .= 'sort='.$col_name;

		//om den är vald
		if ( $this->sort == $col_name )
		{
			$url .= ( $this->order == 'DESC' ) ? '&order=asc' : '&order=desc';
			$class = ( $this->order == 'DESC' ) ? ' class="desc"' : ' class="asc"';
		}
		else
		{
			$url .= '&order=asc';
			$class = '';
		}
		return '<a href="'.$url.'"'.$class.'>'.$caption.'</a>';
	}

	//----------------------------------------------------------------------------
	// LIST: Skapar html-kod för bläddra-knapparna för listor
	// « Föregående  1  2  3  4  5  [6]  7  8  9  10  11  Nästa »
	//----------------------------------------------------------------------------
	public function create_pagenav()
	{
		$offset = $this->offset;
		$limit = $this->limit;
		$total = $this->found_rows;
		$url = $this->url;

		if ( $limit == 0 )
		{
			return '';
		}

		$num_pages = intval(ceil($total/$limit));
		$cur_page = intval($offset/$limit);

		if ( $num_pages < 2 )
		{
			return '';
		}

		$url .= ( strpos($url, '?') === false ) ? '?' : '&';

		//räknar ut första och sista siffran
		$start = ( $cur_page > 5 ) ? ($cur_page-5) : 0;
		$end = ( $cur_page > ($num_pages-5) ) ? $num_pages : ($cur_page+5);

		//fyller ut siffrorna så att det så långt som möjligt är 11 siffror (5+1+5)
		if ( $cur_page < 5 )
		{
			$end = ( $num_pages > 10 ) ? 10 : $num_pages;
		}
		if ( $cur_page > ($num_pages-5) )
		{
			$start = ( ($num_pages-10) < 1 ) ? 0 : ($num_pages-10);
		}

		//genererar html (all + prev)
		$html = '<div class="pagenav">' . "\n" . '<ul>' . "\n";
		$html .= '<li><a href="' . $url . 'limit=0">Visa alla</a></li>' . "\n";
		$html .= ( $offset > 0 ) ? '<li><a href="' . $url . 'offset=' . ($offset-$limit) . '">&laquo; Föregående</a></li>' . "\n" : '<li class="nav_disabled">&laquo; Föregående</li>' . "\n";

		//loopar igenom sidorna och skriver ut siffrorna
		for ($p = $start; $p <= $end; $p++)
		{
			if ( ($p*$limit) < $total )
			{
				$html .= ( $p == $cur_page ) ? '<li class="nav_current">' . ($p+1) . '</li>' . "\n" : '<li><a href="' . $url . 'offset=' . ($p*$limit) . '">' . ($p+1) . '</a></li>' . "\n";
			}
		}

		//genererar html (next)
		$html .= ( ($offset+$limit) < $total ) ? '<li><a href="' . $url . 'offset=' . ($offset+$limit) . '">Nästa &raquo;</a></li>' . "\n" : '<li class="nav_disabled">Nästa &raquo;</li>' . "\n";
		$html .= '</ul>' . "\n" . '</div>' . "\n" . '<div class="clear"></div>' . "\n";

		return $html;
	}
}
//------------------------------------------------------------------------------
// OptiAdmin :: Rad-klassen (view, edit, delete, save)
//------------------------------------------------------------------------------
abstract class OptiRow
{
	//data
	var $id = NULL;
	var $data = array();
	var $prev_id = NULL;
	var $next_id = NULL;

	//params
	var $url = '';
	var $url_up = '';

	//multi & hub
	var $multi_rows = array();
	var $hub_rows = array();

	//----------------------------------------------------------------------------
	// Hämtar id från $_GET['id']
	//----------------------------------------------------------------------------
	public function id_from_get()
	{
		$id = 'id';

		if ( isset($_GET[$id]) && intval($_GET[$id]) > 0 )
		{
			$this->id = intval($_GET[$id]);
		}
		else
		{
			die('error: OptiRow->id_from_get(): '.$id.' saknas');
		}
	}

	//----------------------------------------------------------------------------
	// Hämtar id från $_POST['id']
	//----------------------------------------------------------------------------
	public function id_from_post()
	{
		$id = 'id';

		if ( isset($_POST[$id]) && intval($_POST[$id]) > 0 )
		{
			$this->id = intval($_POST[$id]);
		}
		else
		{
			die('error: OptiRow->id_from_post(): '.$id.' saknas');
		}
	}

	//----------------------------------------------------------------------------
	// VIEW/EDIT: Skapar html-kod för bläddra-knapparna för id:s
	// < ^ >
	//----------------------------------------------------------------------------
	public function create_navbar($list_name)
	{
		global $user;

		//kollar om användaren faktiskt har en lista över id:s
		$id_list = array();
		if ( isset($user->data[$list_name]) )
		{
			$id_list = $user->data[$list_name];
		}
		else
		{
			return '';
		}

		$html = '<div id="navbar">';

		//letar efter aktuellt id
		$key = array_search($this->id, $id_list);
		if ( $key !== false )
		{
			//PREVIOUS
			$prev_id = ( isset($id_list[($key-1)]) ) ? $id_list[($key-1)] : false;
			if ( $prev_id !== false )
			{
				$this->prev_id = $prev_id;
				$html .= '<a class="nav_left" href="'.$this->url.'&id='.$this->prev_id.'" alt="Föregående" title="Föregående"></a>&nbsp;';
			}
			else
			{
				$html .= '<a class="nav_empty href="#"></a>&nbsp;';
			}

			//UP
			$html .= '<a class="nav_up" href="'.$this->url_up.'" alt="Upp" title="Upp"></a>&nbsp;';

			//NEXT
			$next_id = ( isset($id_list[($key+1)]) ) ? $id_list[($key+1)] : false;
			if ( $next_id !== false )
			{
				$this->next_id = $next_id;
				$html .= '<a class="nav_right" href="'.$this->url.'&id='.$this->next_id.'" alt="Nästa" title="Nästa"></a>';
			}
		}
		$html .= '</div>';

		return $html;
	}
}
//------------------------------------------------------------------------------
// OptiAdmin :: Inställnings-klassen ($cfg) (load, get, save, delete)
//------------------------------------------------------------------------------
class OptiConfig
{
	var $data = array();

	//----------------------------------------------------------------------------
	// Laddar inställningar från settings-tabellen (obs! bara en gång)
	//----------------------------------------------------------------------------
	protected function load()
	{
		if ( empty($this->data) )
		{
			$this->data = result(query("SELECT * FROM {$this->settings_table} ORDER BY config_key, config_value, id ASC"));
		}
	}

	//----------------------------------------------------------------------------
	// Hämtar inställning/value för en spec. nyckel/key (tar första bästa)
	//----------------------------------------------------------------------------
	public function get($config_key)
	{
		$this->load();
		foreach ($this->data as $d)
		{
			if ( $d['config_key'] == $config_key )
			{
				return $d['config_value'];
			}
		}
		return NULL;
	}

	//----------------------------------------------------------------------------
	// Sparar en inställning (skriver över ev. gammal)
	//----------------------------------------------------------------------------
	public function save($config_key, $config_value)
	{
		//kollar om värdet redan finns (hämtar id)
		$sql = "SELECT *
						FROM {$this->settings_table}
						WHERE config_key = '$config_key'
						LIMIT 1";
		$result = result(query($sql), false);
		if ( isset($result['id']) && intval($result['id']) > 0 )
		{
			//uppdaterar gammal rad
			$id = intval($result['id']);
			$sql = "UPDATE {$this->settings_table}
							SET config_key = '$config_key',
									config_value = '$config_value'
							WHERE id = $id
							LIMIT 1";
			$result = query($sql);
			return $result;
		}
		else
		{
			//skapar ny rad
			$sql = "INSERT INTO {$this->settings_table}
							SET config_key = '$config_key',
									config_value = '$config_value'";
			$result = query($sql);
			return $result;
		}
	}

	//----------------------------------------------------------------------------
	// Tar bort en spec. inställning (t.ex. en användare)
	//----------------------------------------------------------------------------
	public function delete($config_id)
	{
		global $cfg;

		$sql = "DELETE
						FROM {$cfg->settings_table}
						WHERE id = $config_id
						LIMIT 1";
		$result = query($sql);
		return $result;
	}
}
//------------------------------------------------------------------------------
// OptiAdmin :: Användar-klassen ($user) (login, logout)
//------------------------------------------------------------------------------
class OptiUser
{
	var $id = NULL;
	var $logged_in = false;
	var $data = array();

	//----------------------------------------------------------------------------
	// Kollar om användare är inloggad
	//----------------------------------------------------------------------------
	public function __construct($id = NULL)
	{
		global $cfg, $lang;

		//skapar/fortsätter sessionen
		ini_set('session.use_only_cookies', 1);
		ini_set('session.use_trans_sid', 0);
		session_name($cfg->session_name);
		session_start();

		// (1): Redan inloggad användare (med SESSION)
		if ( isset($_SESSION['logged_in']) && isset($_SESSION[$cfg->secret_key]) && $_SESSION[$cfg->secret_key] === $cfg->secret_val )
		{
			$this->id = $_SESSION['user_id'];
			$this->logged_in = true;
			$this->data = $_SESSION['user_data'];
		}
		// (2): Försöker logga in (med POST)
		elseif ( isset($_POST['username']) && isset($_POST['password']) )
		{
			$login_username = clean($_POST['username']);
			$login_password = clean($_POST['password']);

			//kontrollerar användarnamn och lösenord
			$this->login($login_username, md5($login_password));

			//loggar misslyckade försök
			if ( !$this->logged_in )
			{
				write_log('Misslyckat inloggningsförsök från IP: ['.$_SERVER['REMOTE_ADDR'].'] med användarnamn: ['.$login_username.'] och lösenord: ['.$login_password.']');
				echo $lang->wrong_password;
				die();
			}
		}
		// (3): Har cookie, ladda in och checka password från cookien
		elseif ( isset($_COOKIE[$cfg->cookie_name]) )
		{
			$cookie_getdata = $_COOKIE[$cfg->cookie_name];

			//reverse magic_quotes_gpc/magic_quotes_sybase effects on those vars if ON
			if ( get_magic_quotes_gpc() )
			{
				$cookie_getdata = stripslashes($cookie_getdata);
			}
			$cookie_getdata = unserialize($cookie_getdata);

			if ( !empty($cookie_getdata['u']) && !empty($cookie_getdata['p']) )
			{
				//kontrollerar användarnamn och lösenord (md5)
				$this->login(clean($cookie_getdata['u']), clean($cookie_getdata['p']));
			}
			else
			{
				//dödar sessionen
				session_unset();
				session_destroy();

				//tar bort cookien
				setcookie($cfg->cookie_name, '', time() - 3600);

				die('bad cookie');
			}
		}

		//stänger sessionen (så att andra sidor kan använda session samtidigt)
		session_write_close();

		//om användaren inte är inloggad: redirecta till inloggninsformuläret
		if ( $this->logged_in == false )
		{
			header('Location: '.$cfg->admin_base_filename.'?module=index&action=login');
			die();
		}
	}

	//----------------------------------------------------------------------------
	// Försöker logga in en användare (kollar mot databasen)
	// parametrar: användarnamn, lösenord(md5)
	//----------------------------------------------------------------------------
	protected function login($login_username, $login_password)
	{
		global $cfg;

		//letar i databasen
		$value = $login_username.'|'.$login_password;
		$sql = "SELECT *
						FROM {$cfg->settings_table}
						WHERE config_key = 'user' AND config_value = '$value'
						LIMIT 1";
		$result = result(query($sql), false);
		if ( count($result) > 0 )
		{
			$this->id = intval($result['id']);
			$this->logged_in = true;
			$this->data = array();

			//sparar i session
			$_SESSION['user_id'] = $this->id;
			$_SESSION['logged_in'] = 'yes';
			$_SESSION['user_data'] = $this->data;
			$_SESSION[$cfg->secret_key] = $cfg->secret_val;

			//sparar en cookie med md5(password) + (för autologin!)
			$cookie_setdata = array('u' => $login_username, 'p' => $login_password);
			setcookie($cfg->cookie_name, serialize($cookie_setdata), time() + 31536000);
		}
		else
		{
			//dödar sessionen
			session_unset();
			session_destroy();

			//tar bort cookien
			setcookie($cfg->cookie_name, '', time() - 3600);
		}
	}

	//----------------------------------------------------------------------------
	// Loggar ut användare
	//----------------------------------------------------------------------------
	public function logout()
	{
		global $cfg;

		//skapar/fortsätter sessionen
		ini_set('session.use_only_cookies', 1);
		ini_set('session.use_trans_sid', 0);
		session_name($cfg->session_name);
		session_start();

		//tar bort cookien
		setcookie($cfg->cookie_name, '', time() - 3600);

		$_SESSION['user_id'] = NULL;
		$_SESSION['logged_in'] = NULL;

		//dödar sessionen
		session_unset();
		session_destroy();

		//redirectar användaren så vi slipper ?logout
		header('Location: '.$cfg->admin_base_filename.'?module=index&action=login');
		die();
	}

	//----------------------------------------------------------------------------
	// Skapar en ny användare
	//----------------------------------------------------------------------------
	public function add_user($username, $password)
	{
		global $cfg, $db_link;

		$password = md5($password);

		$config_value = $username.'|'.$password;

		//lägg in användaren i settings-tabellen
		$sql = "INSERT INTO {$cfg->settings_table}
						SET config_key = 'user',
								config_value = '$config_value'";
		$result = query($sql);
		return $result;
	}

	//----------------------------------------------------------------------------
	// Sparar $user->data i SESSION
	//----------------------------------------------------------------------------
	public function save_data()
	{
		//skapar/fortsätter sessionen
		session_start();

		$_SESSION['user_data'] = $this->data;
		session_write_close();
	}
}







//------------------------------------------------------------------------------
// Paroc_usersTable :: Table
//------------------------------------------------------------------------------
class Paroc_usersTable extends OptiTable
{
	//params (default)
	var $sort = 'sortorder';					//sparas i SESSION
	var $order = 'ASC';				//sparas i SESSION
	var $search = NULL;				//sparas i SESSION
	var $filter = array();		//sparas i SESSION

	//settings
	var $sort_cols = array('id','namn','kundnummer','leverantors_id','telenummer','adress','postnummer','ort','user_typ','password','sortorder','active');
	var $search_cols = array('namn','kundnummer','leverantors_id','telenummer','adress','postnummer','ort');
	var $filter_cols = array('user_typ','active');

	//------------------------------------------------------------------------------
	// LIST: Hämtar rader från tabellen (med sort/order/search/filter)
	//------------------------------------------------------------------------------
	public function load_sql()
	{
		//förbereder MySQL-queryn
		$sql_where = array();
		$sql_limit = ( $this->limit > 0 ) ? "LIMIT $this->offset, $this->limit" : '';

		//FILTER: Användar typ
		if ( isset($this->filter['user_typ']) && $this->filter['user_typ'] !== '' )
		{
			$sql_where[] = "FIND_IN_SET('".clean($this->filter['user_typ'])."', user.user_typ) > 0";
			$this->filter_search_message .= '<p>Filtrerar på: <strong>Användar typ</strong></p>';
		}
		//FILTER: Active
		if ( isset($this->filter['active']) && $this->filter['active'] !== '' )
		{
			$sql_where[] = "user.active = ".intval($this->filter['active']);
			$this->filter_search_message .= '<p>Filtrerar på: <strong>Active</strong></p>';
		}

		//SEARCH
		if ( $this->search )
		{
			$sql_where[] = $this->sql_search('user');
			$this->filter_search_message .= '<p>Söker på: <strong>&quot;'.$this->search.'&quot;</strong></p>';
		}

		$this->antal = 0;
		$this->rows = array();
		$this->id_list = array();
		$sql_where = ( count($sql_where) > 0 ) ? "WHERE ".implode(" AND ", $sql_where) : '';
		$sql = "SELECT SQL_CALC_FOUND_ROWS user.*
						FROM paroc_users AS user
						$sql_where
						ORDER BY {$this->sort} {$this->order}
						$sql_limit";
		$result = query($sql);
		while ($r = mysqli_fetch_array($result, MYSQLI_ASSOC))
		{
			$this->rows[intval($r['id'])] = $r;
			$this->id_list[] = intval($r['id']);
			$this->antal++;
		}
		$this->sql_found_rows();
	}

	//------------------------------------------------------------------------------
	// Hämtar alla rader från tabellen (utan sort/order/search/filter)
	//------------------------------------------------------------------------------
	public function load_all($sql_where = '')
	{
		$this->rows = array();
		$sql = "SELECT user.*
						FROM paroc_users AS user
						$sql_where
						ORDER BY user.sortorder ASC";
		$result = query($sql);
		while ($r = mysqli_fetch_array($result, MYSQLI_ASSOC))
		{
			$this->rows[intval($r['id'])] = $r;
		}
	}
}

//------------------------------------------------------------------------------
// Paroc_users :: Row
//------------------------------------------------------------------------------
class Paroc_usersRow extends OptiRow
{
	//----------------------------------------------------------------------------
	// LOAD: Laddar in data från en rad
	//----------------------------------------------------------------------------
	public function load_data()
	{
		$sql = "SELECT user.*
						FROM paroc_users AS user
						WHERE user.id = {$this->id}
						LIMIT 1";
		$result = query($sql);
		$this->data = mysqli_fetch_array($result, MYSQLI_ASSOC);

		if ( empty($this->data) )
		{
			die('error: OptiRow->load_data(): kunde inte hitta raden');
		}
	}

	//----------------------------------------------------------------------------
	// SAVE: Sparar (skapar/uppdaterar) en (ny/gammal) rad
	//----------------------------------------------------------------------------
	public function save($sql_set)
	{
		global $db_link;

		//ny rad (skapa)
		if ( $this->id == 0 )
		{
			//skapar ny rad
			$sql = "INSERT INTO paroc_users
							SET $sql_set";
			$result = query($sql);
			if ( $result )
			{
				$this->id = mysqli_insert_id($db_link);
				//return true;
			}
			else
			{
				die('error: OptiRow->save(): kunde inte skapa ny rad');
			}
		}
		//gammal rad (uppdatera)
		else
		{
			//uppdaterar gammal rad
			$sql = "UPDATE paroc_users
							SET $sql_set
							WHERE id = {$this->id}
							LIMIT 1";
			$result = query($sql);
			if ( $result )
			{
				//return true;
			}
			else
			{
				die('error: OptiRow->save(): kunde inte uppdatera rad');
			}
		}

		return true;
	}

	//----------------------------------------------------------------------------
	// DELETE: Tar bort en rad (samt alla ev. multi+hub)
	//----------------------------------------------------------------------------
	public function delete()
	{
		//tar bort raden
		$sql = "DELETE FROM paroc_users
						WHERE id = {$this->id}
						LIMIT 1";
		$result = query($sql);

		return $result;
	}

	//----------------------------------------------------------------------------
	// COPY: Kopierar en rad (samt alla ev. multi+hub)
	//----------------------------------------------------------------------------
	public function copy()
	{
		global $db_link;

		//kopierar raden
		$copy_id = 0;
		$sql = "INSERT INTO paroc_users(namn,kundnummer,leverantors_id,telenummer,adress,postnummer,ort,user_typ,password,sortorder,active)
						SELECT namn,kundnummer,leverantors_id,telenummer,adress,postnummer,ort,user_typ,password,sortorder,active
						FROM paroc_users
						WHERE id = {$this->id}
						ORDER BY id ASC";
		$result = query($sql);
		if ( $result )
		{
			$copy_id = mysqli_insert_id($db_link);
		}

		return $copy_id;
	}
}



//------------------------------------------------------------------------------
// ProductsTable :: Table
//------------------------------------------------------------------------------
class ProductsTable extends OptiTable
{
	//params (default)
	var $sort = 'sortorder';					//sparas i SESSION
	var $order = 'ASC';				//sparas i SESSION
	var $search = NULL;				//sparas i SESSION
	var $filter = array();		//sparas i SESSION

	//settings
	var $sort_cols = array('id','name','typ','vikt_sack','sack_kubik','sattning','min_densitet','lambavarde','sortorder','active');
	var $search_cols = array('name');
	var $filter_cols = array('typ','active');

	//------------------------------------------------------------------------------
	// LIST: Hämtar rader från tabellen (med sort/order/search/filter)
	//------------------------------------------------------------------------------
	public function load_sql()
	{
		//förbereder MySQL-queryn
		$sql_where = array();
		$sql_limit = ( $this->limit > 0 ) ? "LIMIT $this->offset, $this->limit" : '';

		//FILTER: Typ
		if ( isset($this->filter['typ']) && $this->filter['typ'] !== '' )
		{
			$sql_where[] = "FIND_IN_SET('".clean($this->filter['typ'])."', prod.typ) > 0";
			$this->filter_search_message .= '<p>Filtrerar på: <strong>Typ</strong></p>';
		}
		//FILTER: Active
		if ( isset($this->filter['active']) && $this->filter['active'] !== '' )
		{
			$sql_where[] = "prod.active = ".intval($this->filter['active']);
			$this->filter_search_message .= '<p>Filtrerar på: <strong>Active</strong></p>';
		}

		//SEARCH
		if ( $this->search )
		{
			$sql_where[] = $this->sql_search('prod');
			$this->filter_search_message .= '<p>Söker på: <strong>&quot;'.$this->search.'&quot;</strong></p>';
		}

		$this->antal = 0;
		$this->rows = array();
		$this->id_list = array();
		$sql_where = ( count($sql_where) > 0 ) ? "WHERE ".implode(" AND ", $sql_where) : '';
		$sql = "SELECT SQL_CALC_FOUND_ROWS prod.*
						FROM products AS prod
						$sql_where
						ORDER BY {$this->sort} {$this->order}
						$sql_limit";
		$result = query($sql);
		while ($r = mysqli_fetch_array($result, MYSQLI_ASSOC))
		{
			$this->rows[intval($r['id'])] = $r;
			$this->id_list[] = intval($r['id']);
			$this->antal++;
		}
		$this->sql_found_rows();
	}

	//------------------------------------------------------------------------------
	// Hämtar alla rader från tabellen (utan sort/order/search/filter)
	//------------------------------------------------------------------------------
	public function load_all($sql_where = '')
	{
		$this->rows = array();
		$sql = "SELECT prod.*
						FROM products AS prod
						$sql_where
						ORDER BY prod.sortorder ASC";
		$result = query($sql);
		while ($r = mysqli_fetch_array($result, MYSQLI_ASSOC))
		{
			$this->rows[intval($r['id'])] = $r;
		}
	}
}

//------------------------------------------------------------------------------
// Products :: Row
//------------------------------------------------------------------------------
class ProductsRow extends OptiRow
{
	//----------------------------------------------------------------------------
	// LOAD: Laddar in data från en rad
	//----------------------------------------------------------------------------
	public function load_data()
	{
		$sql = "SELECT prod.*
						FROM products AS prod
						WHERE prod.id = {$this->id}
						LIMIT 1";
		$result = query($sql);
		$this->data = mysqli_fetch_array($result, MYSQLI_ASSOC);

		if ( empty($this->data) )
		{
			die('error: OptiRow->load_data(): kunde inte hitta raden');
		}
	}

	//----------------------------------------------------------------------------
	// SAVE: Sparar (skapar/uppdaterar) en (ny/gammal) rad
	//----------------------------------------------------------------------------
	public function save($sql_set)
	{
		global $db_link;

		//ny rad (skapa)
		if ( $this->id == 0 )
		{
			//skapar ny rad
			$sql = "INSERT INTO products
							SET $sql_set";
			$result = query($sql);
			if ( $result )
			{
				$this->id = mysqli_insert_id($db_link);
				//return true;
			}
			else
			{
				die('error: OptiRow->save(): kunde inte skapa ny rad');
			}
		}
		//gammal rad (uppdatera)
		else
		{
			//uppdaterar gammal rad
			$sql = "UPDATE products
							SET $sql_set
							WHERE id = {$this->id}
							LIMIT 1";
			$result = query($sql);
			if ( $result )
			{
				//return true;
			}
			else
			{
				die('error: OptiRow->save(): kunde inte uppdatera rad');
			}
		}

		return true;
	}

	//----------------------------------------------------------------------------
	// DELETE: Tar bort en rad (samt alla ev. multi+hub)
	//----------------------------------------------------------------------------
	public function delete()
	{
		//tar bort raden
		$sql = "DELETE FROM products
						WHERE id = {$this->id}
						LIMIT 1";
		$result = query($sql);

		return $result;
	}

	//----------------------------------------------------------------------------
	// COPY: Kopierar en rad (samt alla ev. multi+hub)
	//----------------------------------------------------------------------------
	public function copy()
	{
		global $db_link;

		//kopierar raden
		$copy_id = 0;
		$sql = "INSERT INTO products(name,typ,vikt_sack,sack_kubik,sattning,min_densitet,lambavarde,sortorder,active)
						SELECT name,typ,vikt_sack,sack_kubik,sattning,min_densitet,lambavarde,sortorder,active
						FROM products
						WHERE id = {$this->id}
						ORDER BY id ASC";
		$result = query($sql);
		if ( $result )
		{
			$copy_id = mysqli_insert_id($db_link);
		}

		return $copy_id;
	}
}










//------------------------------------------------------------------------------
// OptiAdmin :: Hjälp-klass med diverse funktioner
//------------------------------------------------------------------------------
class OptiAdmin
{
	//----------------------------------------------------------------------------
	// VIEW :: MySQL till HTML
	//----------------------------------------------------------------------------
	public static function view($type, $value, $col_name = '')
	{
		global $cfg;

		//$r = return-värdet, $value = original-MySQL-värdet
		$r = $value;

		switch ($type)
		{
			case 'int':
				$r = ( $value != '' ) ? intval($value) : '';
				break;

			case 'float':
				$r = ( $value != '' ) ? floatval($value) : '';
				break;

			case 'text':
				//konverterar specialtecken & " ' < > till html-koder
				$r = htmlspecialchars($r, ENT_QUOTES);
				break;

			case 'textarea':
				//konverterar specialtecken & " ' < > till html-koder
				$r = htmlspecialchars($r, ENT_QUOTES);
				//omvandlar radbrytningar till <br />
				$r = nl2br($r);
				break;

			case 'html':
				$r = $value;
				break;

			case 'date':
				$r = $value;
				break;

			case 'datetime':
				$r = $value;
				break;

			case 'money':
				$r = $value;
				break;

			case 'bool':
				$r = ( intval($r) == 1 ) ? '<span class="bool_true"></span>' : '<span class="bool_false"></span>';
				break;

			case 'file':
				$r = ( $value ) ? '<a target="_blank" href="'.$cfg->upload_files_path.$value.'">'.$value.'</a>' : '';
				//skriv ut varning om filen inte finns
				if ( $value && !file_exists($cfg->upload_files_path.$value) )
				{
					$r = '<span class="icon_warning">Kan inte hitta filen:</span> <a target="_blank" href="'.$cfg->upload_files_path.$value.'">'.$value.'</a>';
				}
				break;

			case 'image':
				$r = ( $value ) ? '<a class="lightboxed" rel="highslide" href="'.$cfg->upload_images_path.$value.'" title="'.$value.'"><img src="'.$cfg->upload_thumbs_path.$value.'" /></a>' : '';
				//skriv ut varning om filen inte finns
				if ( $value && !file_exists($cfg->upload_images_path.$value) )
				{
					$r = '<span class="icon_warning">Kan inte hitta filen:</span> <a target="_blank" href="'.$cfg->upload_images_path.$value.'">'.$value.'</a>';
				}
				break;

			case 'select':
				//hämtar titel
				$r = ( $value && isset($cfg->select_data[$col_name][$value]) ) ? $cfg->select_data[$col_name][$value] : $value;
				break;

			case 'selectmultiple':
				//konverterar från sträng 'a,b,c' till sträng med titlar 'Aa, Bb, Cc'
				$multiple = explode(',', $value);
				foreach ($multiple as &$m)
				{
					$m = ( $m && isset($cfg->select_data[$col_name][$m]) ) ? $cfg->select_data[$col_name][$m] : $m;
				}
				unset($m);
				$r = implode(', ', $multiple);
				break;

			case 'foreign':
				//konverterar specialtecken & " ' < > till html-koder
				$r = htmlspecialchars($r, ENT_QUOTES);
				$r = ( $r ) ? '<span class="foreign">'.$r.'</span>' : '';
				break;

			case 'sort':
				$r = ( $value != '' ) ? '<span class="sort">'.intval($value).'</span>' : '';
				break;

			case 'geoposition':
				$r = ( $r ) ? '<a class="geoposition" href="'.$cfg->admin_base_filename.'?module=index&action=view_geoposition&geopos='.$r.'" onclick="return hs.htmlExpand(this, { objectType: \'iframe\' } )" title="'.$r.'"></a>' : '';
				break;
		}

		return $r;
	}

	//----------------------------------------------------------------------------
	// EDIT :: MySQL till <INPUT>
	//----------------------------------------------------------------------------
	public static function edit($type, $value, $col_name = '', $allow_empty = false)
	{
		global $cfg;

		//$r = return-värdet, $value = original-MySQL-värdet
		$r = $value;

		switch ($type)
		{
			case 'int':
				$r = '<input type="text" name="'.$col_name.'" value="'.intval($r).'" class="input_'.$type.'" />';
				break;

			case 'float':
			case 'money':
				$r = '<input type="text" name="'.$col_name.'" value="'.floatval($r).'" class="input_'.$type.'" />';
				break;

			case 'text':
				//konverterar specialtecken:   & " ' < >   till motsvarande html-koder:   &amp; &quot; &#039; &lt; &gt;
				$r = '<input type="text" name="'.$col_name.'" value="'.htmlspecialchars($value, ENT_QUOTES).'" class="input_'.$type.'" />';
				break;

			case 'textarea':
				$r = '<textarea name="'.$col_name.'" class="input_'.$type.'">'.$r.'</textarea>';
				break;

			case 'html':
				$r = '<p><span class="link icon_htmleditor" onclick="tinyMCE.execCommand(\'mceToggleEditor\', false, \'tinymce_'.$col_name.'\');">visa/göm HTML-redigering</span></p>';
				$r .= '<textarea id="tinymce_'.$col_name.'" name="'.$col_name.'" class="input_'.$type.'">'.str_replace('&', '&amp;', $value).'</textarea>';
				break;

			case 'date':
			case 'datetime':
				$r = '<input id="datepicker_'.$col_name.'" type="text" name="'.$col_name.'" value="'.$value.'" class="input_'.$type.'" />';
				$r .= '&nbsp; <span id="datepicker_'.$col_name.'_toggler" class="link icon_datepicker"></span>';
				$r .= "\n".'<script type="text/javascript">
window.addEvent(\'domready\', function() {
	create_datepicker(\''.$col_name.'\', false);
});
</script>';
				break;

			case 'filter_date':
				//delar upp strängen i start- och slut-datum
				$temp = explode(' - ', $value);
				$date_start = ( isset($temp[0]) && $temp[0] !== '' ) ? date('Y-m-d', strtotime($temp[0])) : NULL;
				$date_end = ( isset($temp[1]) && $temp[1] !== '' ) ? date('Y-m-d', strtotime($temp[1])) : NULL;

				//skapar ett osynligt input-fält + två synliga för start resp. slut
				$r = '<input id="datepicker_'.$col_name.'" type="hidden" name="'.$col_name.'" value="'.$value.'" style="width: 250px;" />';
				$r .= '<input id="datepicker_'.$col_name.'_start" type="text" name="'.$col_name.'_start" value="'.$date_start.'" class="input_date" />';
				$r .= ' <span id="datepicker_'.$col_name.'_start_toggler" class="link icon_datepicker"></span>';
				$r .= 'till&nbsp;<input id="datepicker_'.$col_name.'_end" type="text" name="'.$col_name.'_end" value="'.$date_end.'" class="input_date" />';
				$r .= ' <span id="datepicker_'.$col_name.'_end_toggler" class="link icon_datepicker"></span>';
				$r .= "\n".'<script type="text/javascript">
window.addEvent(\'domready\', function() {
	create_datepicker(\''.$col_name.'\', true);
});
</script>';
				break;

			case 'bool':
				//om vi ska tillåta tom: dvs. gör en selectbox (för filter)
				if ( $allow_empty )
				{
					$value = ( $value === '' ) ? '' : intval($value);
					$r = OptiAdmin::create_select(array(0 => 'Av', 1 => 'På'), $col_name, $value, $allow_empty);
				}
				//annars vanligt: checkbox (för add/edit)
				else
				{
					$checked = ( intval($r) == 1 ) ? ' checked="checked"' : '';
					$r = '<input type="checkbox" name="'.$col_name.'" value="1"'.$checked.' class="input_'.$type.'" />';
				}
				break;

			case 'file':
			case 'image':
				//visa den gamla filen/bilden
				$r = OptiAdmin::view($type, $value, $col_name);
				$r .= '<br />';
				$r .= '<input type="hidden" name="'.$col_name.'" value="'.$value.'" />';
				$r .= '<input type="file" name="'.$col_name.SUFFIX_UPLOADED_FILE.'" size="40" class="input_'.$type.'" />';
				break;

			case 'select':
				$r = OptiAdmin::create_select($cfg->select_data[$col_name], $col_name, $value, $allow_empty);
				break;

			case 'selectmultiple':
				//konverterar från sträng 'a,b,c' till array('a','b','c')
				$selected = explode(',', $value);
				$r = OptiAdmin::create_selectmultiple($cfg->select_data[$col_name], $col_name, $selected, $allow_empty);
				break;

			case 'foreign':
				$data = array();
				$r = OptiAdmin::create_select($data, $col_name, $value, $allow_empty);
				break;

			case 'sort':
				//går inte att redigera
				$r = '<span class="sort">'.intval($r).'</span><input type="hidden" name="'.$col_name.'" value="'.intval($r).'" />';
				break;

			case 'geoposition':
				$r = '<p><span class="link icon_geoeditor" onclick="open_geoposition_map(\''.$col_name.'\', \''.$cfg->geoposition_input_gata.'\', \''.$cfg->geoposition_input_postnr.'\', \''.$cfg->geoposition_input_ort.'\');">öppna karta</span></p>';
				$r .= '<input type="text" name="'.$col_name.'" value="'.$value.'" class="input_'.$type.'" />';
				break;
		}

		return $r;
	}

	//----------------------------------------------------------------------------
	// CLEAN :: POST till MySQL
	//----------------------------------------------------------------------------
	public static function clean($type, $col_name)
	{
		global $cfg, $db_link;

		//hämtar värde från POST
		$value = ( isset($_POST[$col_name]) ) ? $_POST[$col_name] : NULL;

		//$r = return-värdet, $value = original-POST-värdet
		$r = $value;

		//reverse magic_quotes_gpc/magic_quotes_sybase effects on those vars if ON
	  if ( get_magic_quotes_gpc() && !is_array($r) )
		{
			$r = stripslashes($r);
		}

		switch ($type)
		{
			case 'int':
				$r = intval($r);
				break;

			case 'float':
			case 'money':
				//omvandlar komma till punkt
				$r = floatval(str_replace(',', '.', $r));
				break;

			case 'text':
			  //rensar alla HTML- och PHP-koder från strängen
				$r = strip_tags($r);
				//omvandlar radbrytningar \n, \r och tabbar \t till mellanslag
				$r = str_replace("\n", " ", $r);
				$r = str_replace("\r", " ", $r);
				$r = str_replace("\t", " ", $r);
				break;

			case 'textarea':
			  //rensar alla HTML- och PHP-koder från strängen
				$r = strip_tags($r);
				break;

			case 'html':
				$r = $r;
				break;

			case 'date':
				$r = $r;
				break;

			case 'datetime':
				$r = $r;
				break;

			case 'bool':
				$r = $r;
				break;

			case 'file':
				//kollar om vi har en uppladdad fil och flyttar den
				$new_file = OptiAdmin::upload_file($col_name, $cfg->upload_files_path);
				if ( $new_file )
				{
					//sparar det nya filnamnet
					$r = $new_file;
				}
				break;

			case 'image':
				//kollar om vi har en uppladdad fil och flyttar den
				$new_file = OptiAdmin::upload_file($col_name, $cfg->upload_images_path);
				if ( $new_file )
				{
					//sparar det nya filnamnet
					$r = $new_file;

					//kollar om bilden är för stor (d.v.s. om vi ska storleksändra bilden eller inte)
					if ( OptiAdmin::check_image_size($cfg->upload_images_path.$new_file, $cfg->max_image_width, $cfg->max_image_height) )
					{
						//storleksändrar bilden (skriver över originalet)
						OptiAdmin::create_resized_image($cfg->upload_images_path.$new_file, $cfg->upload_images_path.$new_file, $cfg->max_image_width, $cfg->max_image_height);
					}

					//skapar tumnagel
					OptiAdmin::create_resized_image($cfg->upload_images_path.$new_file, $cfg->upload_thumbs_path.$new_file, $cfg->thumb_width, $cfg->thumb_height);
				}
				break;

			case 'select':
				$r = $r;
				break;

			case 'selectmultiple':
				//konverterar från array('a','b','c') till sträng 'a,b,c'
				$r = ( is_array($r) ) ? implode(',', $r) : NULL;
				break;

			case 'foreign':
				$r = intval($r);
				break;

			case 'sort':
				$r = intval($r);
				break;

			case 'geoposition':
			  //rensar alla HTML- och PHP-koder från strängen
				$r = strip_tags($r);
				break;
		}

		//escapar strängen för INSERT/UPDATE i databas
		$r = mysqli_real_escape_string($db_link, $r);

		return $r;
	}

	//----------------------------------------------------------------------------
	// Kollar om vi har en uppladdad fil och flyttar den
	// Returnerar det nya filnamnet eller false om ingen fil fanns
	//----------------------------------------------------------------------------
	public static function upload_file($col_name, $destination_path, $overwrite_file = false)
	{
		//om vi har en uppladdad fil (och filen verkligen är uppladdad med HTTP POST!)
		if ( isset($_FILES[$col_name.SUFFIX_UPLOADED_FILE]) && is_uploaded_file($_FILES[$col_name.SUFFIX_UPLOADED_FILE]['tmp_name']) )
		{
			$file = $_FILES[$col_name.SUFFIX_UPLOADED_FILE];
			//om vi inte har några errors och vi har en fil större än 0 bytes
			if ( !$file['error'] && $file['size'] > 0 )
			{
				//skapar ett nytt (webbsäkert!) filnamn och ny sökväg
				$new_filename = OptiAdmin::clean_filename($file['name']);

				//om vi inte får skriva över gamla filer:
				if ( $overwrite_file == false )
				{
					//kolla om filen redan finns
					if ( file_exists($destination_path.$new_filename) )
					{
						//starta en loop där vi försöker hitta ett nytt filnamn som inte finns sen tidigare
						// my_file.jpg => my_file_v1.jpg => my_file_v2.jpg => osv...
						$file_info = pathinfo($destination_path.$new_filename);
						$i = 1;
						$check_name = $file_info['filename'].'_v'.$i.'.'.$file_info['extension'];
						while ( file_exists($destination_path.$check_name) )
						{
							$check_name = $file_info['filename'].'_v'.$i.'.'.$file_info['extension'];		//['filename'] since PHP 5.2.0
							$i++;
						}
						$new_filename = $check_name;
					}
				}

				//flyttar filen
				if ( move_uploaded_file($file['tmp_name'], $destination_path.$new_filename) )
				{
					return $new_filename;
				}
				else
				{
					die('error: kunde inte flytta den uppladdade filen');
				}
			}
			else
			{
				die('error: ett fel uppstod när filen skulle laddas upp');
			}
		}

		return false;
	}

	//----------------------------------------------------------------------------
	// Kollar om vi ska storleksändra bilden
	// Returnerar true om bilden är större än angivna storlekar (bredd eller höjd), annars false
	//----------------------------------------------------------------------------
	public static function check_image_size($source_file, $max_width, $max_height)
	{
		//max_width och max_height måste vara större än 0
		if ( $max_width > 0 && $max_height > 0 )
		{
			//läser in originalbildens bild-dimensioner och kollar om den är för bred eller för hög
			list($source_width, $source_height) = getimagesize($source_file);
			if ( $source_width > $max_width || $source_height > $max_height )
			{
				return true;
			}
			else
			{
				return false;
			}
		}
		else
		{
			return false;
		}
	}

	//----------------------------------------------------------------------------
	// Skapar en "tumnagel" (förminskad kopia) utifrån en bild
	// OBS! skriver över ev. befintlig fil med samma sökväg som $resized_file
	// Returnerar true om allt går bra, annars die()
	//----------------------------------------------------------------------------
	public static function create_resized_image($source_file, $resized_file, $width, $height)
	{
		//läser in originalbildens bild-dimensioner+bildtyp och räknar ut "tumnagelns" storlek
		list($source_width, $source_height, $imagetype) = getimagesize($source_file);
		$ratio = ($source_width / $source_height);
		if ( ($width / $height) > $ratio )
		{
			//stående porträtt
			$width = intval($height * $ratio);
		}
		else
		{
			//liggande landskap
			$height = intval($width / $ratio);
		}

		//läser in originalbilden (JPG, PNG eller GIF)
		if ( $imagetype == IMAGETYPE_JPEG )
		{
			$source = imagecreatefromjpeg($source_file);
		}
		elseif ( $imagetype == IMAGETYPE_PNG )
		{
			$source = imagecreatefrompng($source_file);
		}
		elseif ( $imagetype == IMAGETYPE_GIF )
		{
			$source = imagecreatefromgif($source_file);
		}
		else
		{
			die('error: OptiAdmin::resize_image(): bildformatet stöds ej');
		}

		//skapar den nya "tumnageln"
		$resized = imagecreatetruecolor($width, $height);

		//ändra storlek
		if ( !imagecopyresampled($resized, $source, 0, 0, 0, 0, $width, $height, $source_width, $source_height) )
		{
			die('error: OptiAdmin::resize_image(): kunde inte ändra storlek på bilden');
		}

		//sparar "tumnageln" (JPG, PNG eller GIF)
		if ( $imagetype == IMAGETYPE_JPEG )
		{
			$result = imagejpeg($resized, $resized_file, 90);
		}
		elseif ( $imagetype == IMAGETYPE_PNG )
		{
			$result = imagepng($resized, $resized_file);
		}
		elseif ( $imagetype == IMAGETYPE_GIF )
		{
			$result = imagegif($resized, $resized_file);
		}

		if ( $result )
		{
			return true;
		}
		else
		{
			die('error: OptiAdmin::resize_image(): kunde inte spara filen');
		}
	}

	//----------------------------------------------------------------------------
	// Skapar <select>
	//----------------------------------------------------------------------------
	public static function create_select(array $data, $name, $selected = '', $allow_empty = false)
	{
		global $cfg;

		$html = '<select name="'.$name.'">';
		$html .= ( $allow_empty ) ? '<option value=""></option>' : '';
		foreach ($data as $key => $caption)
		{
			$html .= ( $selected !== '' && $key == $selected ) ? '<option value="'.$key.'" selected="selected">'.$caption.'</option>' : '<option value="'.$key.'">'.$caption.'</option>';
		}
		return $html.'</select>';
	}

	//----------------------------------------------------------------------------
	// Skapar <select multiple="multiple">
	//----------------------------------------------------------------------------
	public static function create_selectmultiple(array $data, $name, $selected = array(), $allow_empty = false)
	{
		global $cfg;

		$html = '<select name="'.$name.'[]" multiple="multiple" size="'.count($data).'" >';
		$html .= ( $allow_empty ) ? '<option value=""></option>' : '';
		foreach ($data as $key => $caption)
		{
			$html .= ( in_array($key, $selected) ) ? '<option value="'.$key.'" selected="selected">'.$caption.'</option>' : '<option value="'.$key.'">'.$caption.'</option>';
		}
		return $html.'</select>';
	}

	//------------------------------------------------------------------------------
	// Webbsäkrar ett filnamn ('Abc: Källe & Åda' => 'abc-kaelle-ada.jpg')
	//------------------------------------------------------------------------------
	public static function clean_filename($f)
	{
		//byter ut (å-ä-ö => a-ae-oe)
		$f = str_replace('å', 'a', $f);
		$f = str_replace('ä', 'ae', $f);
		$f = str_replace('ö', 'oe', $f);

    $f = utf8_decode($f);

		//byter ut konstiga bokstäver (åäö => aao)
    $a = 'ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõöøùúûýýþÿŔŕ';
    $b = 'aaaaaaaceeeeiiiidnoooooouuuuybsaaaaaaaceeeeiiiidnoooooouuuyybyrr';
    $f = strtr($f, utf8_decode($a), $b);

		//lowercase
    $f = strtolower($f);

		//mellanslag och _ till -
		$f = str_replace('_', '-', $f);
		$f = str_replace(' ', '-', $f);

		//tar bara med tillåtna tecken: a-z 0-9 - .
    $f = preg_replace("([^a-z0-9-\.])", "", $f);

		//tar bort dubbla -- => -
    $f = preg_replace("(-{2,})", "-", $f);

    $f = utf8_encode($f);

		return $f;
	}

	//------------------------------------------------------------------------------
	// Formaterar tid från "2009-06-12 20:42:05" till "idag kl. 20:42:35", "igår kl. 20:42" eller "12 jun kl. 20:42"
	//------------------------------------------------------------------------------
	public static function format_datetime($datetime)
	{
		global $lang;

		if ( !$datetime || $datetime == '0000-00-00 00:00:00' )
		{
			return '-';
		}

		$date = substr($datetime, 0, 10);
		$year = substr($datetime, 0, 4);
		$month = intval(substr($datetime, 5, 2));
		$day = intval(substr($datetime, 8, 2));
		$time = substr($datetime, 11, 5);

		$today = date('Y-m-d');

		//idag (+tid +sek)
		if ( $date === $today )
		{
			//ta även med sekunder här!
			return $lang->today.' '.substr($datetime, 11, 8);
		}
		//igår (+tid)
		elseif ( $date === date('Y-m-d', strtotime('-1 day')) )
		{
			return $lang->yesterday.' '.$time;
		}
		//den här månaden och året (+tid)
		elseif ( substr($datetime, 0, 7) === substr($today, 0, 7) )
		{
			return $day.' '.strtolower($lang->shortmonth_names[$month]).' '.$time;
		}
		//det här året
		elseif ( $year === substr($today, 0, 4) )
		{
			return $day.' '.strtolower($lang->shortmonth_names[$month]);
		}
		//annars
		else
		{
			return $day.' '.strtolower($lang->shortmonth_names[$month]).' '.$year;
		}
	}

	//------------------------------------------------------------------------------
	// Formaterar datum från "2009-06-12" till "idag", "igår" eller "mån 12 jun"
	//------------------------------------------------------------------------------
	public static function format_datum($date)
	{
		global $lang;

		if ( !$date || $date == '0000-00-00' )
		{
			return '-';
		}

		$date = substr($date, 0, 10);
		$month = intval(substr($date, 5, 2));
		$day = intval(substr($date, 8, 2));

		$today = date('Y-m-d');

		if ( $date == $today )
		{
			return $lang->today;
		}
		elseif ( $date == date('Y-m-d', strtotime('-1 day')) )
		{
			return $lang->yesterday;
		}
		else
		{
			$year = ( substr($date, 0, 4) != substr($today, 0, 4) ) ? ' '.substr($date, 0, 4) : '';
			return $day.' '.strtolower($lang->shortmonth_names[$month]).$year;
		}
	}

	//----------------------------------------------------------------------------
	// Sparar sortering på en spec. tabell
	//----------------------------------------------------------------------------
	public static function sort_table($table, $primary_key, $sortcol, $order)
	{
		foreach ($order as $key => $value)
		{
			$sortnr = intval($key);
			//plussar på sortcol så att den börjar på 1 (lättare att skilja från nya rader som får 0)
			$sortnr++;
			$temp = explode('_', $value);
			$id = intval($temp[(count($temp)-1)]);

			if ( $id > 0 )
			{
				$query = "UPDATE $table
									SET $sortcol = $sortnr
									WHERE $primary_key = $id
									LIMIT 1";
				$result = query($query);
			}
		}
	}

	//----------------------------------------------------------------------------
	// Kollar efter foreign-children
	//----------------------------------------------------------------------------
	public static function check_foreign_children($fk_id, $child_pk, $child_table, $child_caption, $child_module, $child_col)
	{
		$sql = "SELECT $child_pk
						FROM $child_table
						WHERE $child_col = $fk_id";
		$info = result(query($sql));
		if ( count($info) > 0 )
		{
			echo 'Den här raden kan inte tas bort eftersom den används '.count($info).' gånger i <a href="?module='.$child_module.'&action=list">'.$child_caption.'</a>';
			die();
		}
	}
}

?>
